import path from 'path';
import fs from 'fs';
import mkdirp from 'mkdirp';
import winston from 'winston';

/**
 * Downloads CVG (poly and raster) data.
 */
export default class FetchCvgJob {
	constructor(config, workspaceDir, sftpAgent) {
		this.config = config;
		this.workspaceDir = workspaceDir;
		this.sftpAgent = sftpAgent;
	}

	async run() {
		const processor = new RootDirectoryProcessor(
			this.config,
			this.workspaceDir,
			this.sftpAgent,
			'',
			null,
			{}
		);
		const result = await processor.run();

		winston.info('FetchCvgJobResult:\n' + JSON.stringify(result, null, 2), {
			label: 'jobResult',
			job: this.constructor.name,
			result,
			timestamp: new Date(),
		});

		return result;
	}
}

/** Processes single file or directory. */
class BaseFileProcessor {
	constructor(config, workspaceDir, sftpAgent, remotePath, file, logMeta) {
		Object.assign(this, { config, workspaceDir, sftpAgent, remotePath, file, logMeta });
	}

	get taskName() {
		return this.constructor.name;
	}

	async run() {
		return {
			errors: [],
			warnings: [],
		};
	}	
}

/**
 * Processes directory by recursively creating and running processors for each file in it.
 *
 * Child classes should override createSubProcessor method to create appropriate processor for given file.
 * If that method returns null, the file will be ignored.
 */
class RecursiveDirectoryProcessor extends BaseFileProcessor {
	async run() {
		const result = await super.run();
		result.tasks = {};
		result.ignored = [];

		const files = await this.sftpAgent.list(this.remotePath);
		this.sortFiles(files);

		mkdirp(this.workspaceDir);

		for (const file of files) {
			const subRemotePath = `${this.remotePath}/${file.name}`;
			const defaultSubWorkspaceDir = path.join(this.workspaceDir, file.name);
			const sub = this.createSubProcessor(file, subRemotePath, defaultSubWorkspaceDir);
			if (sub) {
				winston.debug(`Will use ${sub.constructor.name} to process ${subRemotePath}`);
				await sub.run().then(
					subResult => {
						result.tasks[sub.taskName] = subResult;
					},
					error => {
						winston.error(`Error in ${sub.constructor.name} for ${subRemotePath}: ${error}`, {
							...this.logMeta,
							processor: sub.constructor.name,
							remotePath: subRemotePath,
						});
						result.errors.push({
							processor: sub.constructor.name,
							remotePath: subRemotePath,
							timestamp: new Date(),
							error: error.message,
						});
					}
				);
			} else {
				winston.info(
					`Ignoring file ${file.name} in ${this.remotePath || 'the root directory'}`,
					{
						...this.logMeta,
						remotePath: subRemotePath
					}
				);
				result.ignored.push(subRemotePath);
			}
		}

		return result;
	}

	sortFiles(files) {
		// no-op by default
	}

	createSubProcessor(file, remotePath, defaultWorkspaceDir) {
		return null;
	}
}

/**
 * Processor for the root directory. Ignores everything except 'poly' and 'raster'
 * subdirectories, as specified in config.
 */
class RootDirectoryProcessor extends RecursiveDirectoryProcessor {
	createSubProcessor(file, remotePath, defaultSubWorkspaceDir) {
		let subClass;
		if (file.name === this.config.get('polyDir')) {
			subClass = PolyTopLevelDirectoryProcessor;
		} else if (file.name === this.config.get('rasterDir')) {
			subClass = RasterTopLevelDirectoryProcessor;
		} else {
			return null;
		}

		return new subClass(
			this.config,
			this.workspaceDir,
			this.sftpAgent,
			remotePath,
			file,
			this.logMeta,
		);
	}
}

/**
 * Base class for poly/raster top-level directories. Ignores everything
 * except topDirs specified in config (ATLP_GSM and such). For each topDir, creates subprocessor
 * specified by subProcessorClass instance property.
 */
class TopLevelDirectoryProcessor extends RecursiveDirectoryProcessor {
	createSubProcessor(file, remotePath, defaultSubWorkspaceDir) {
		if (this.config.get('topDirs').indexOf(file.name) >= 0) {
			return new this.subProcessorClass(
				this.config,
				defaultSubWorkspaceDir,
				this.sftpAgent,
				remotePath,
				file,
				{
					...this.logMeta,
					topDir: file.name,
				}
			);
		} else {
			return null;
		}
	}
}

class PolyTopLevelDirectoryProcessor extends TopLevelDirectoryProcessor {
	get taskName() {
		return 'poly';
	}

	subProcessorClass = PolyLevel2DirectoryProcessor;
}

class RasterTopLevelDirectoryProcessor extends TopLevelDirectoryProcessor {
	get taskName() {
		return 'raster';
	}
	
	subProcessorClass = RasterLevel2DirectoryProcessor;
}

/**
 * Base class for level 2 directories. Processes subdirectories with filenames "YYMMDD-SUFFIX",
 * ignores everything else. Allowed suffixes are specified in config.
 * Subprocessor class is specified with subProcessorClass instance property.
 */
class Level2DirectoryProcessor extends RecursiveDirectoryProcessor {
	constructor(config, workspaceDir, sftpAgent, remotePath, file, logMeta) {
		super(config, workspaceDir, sftpAgent, remotePath, file, logMeta);
		this.suffixSeen = {};
	}

	async run() {
		const result = await super.run();
		// after normal processing, check that at least on subdirectory
		// was found for each suffix
		for (const suffix of this.config.get('subDirs')) {
			if (!this.suffixSeen[suffix]) {
				winston.warn(
					`${this.constructor.name} found no files ` +
					`with suffix ${suffix} in ${this.remotePath}`,
					{
						...this.logMeta,
						processor: this.constructor.name,
						label: 'dirFound',
						remotePath: this.remotePath,
						suffix,
					}
				);
				result.warnings.push({
					processor: this.constructor.name,
					remotePath: this.remotePath,
					timestamp: new Date(),
					suffix,
					warning: `No files with suffix ${suffix} in ${this.remotePath}`
				});
			}
		}
		return result;
	}

	/**
	 * Ignore everything without supported suffix. When the suffix is supported, 
	 * ignore the directory anyway if we already seen that suffix - files are
	 * processed in descending alphabetical order, so the newest directory with
	 * given suffix is processed, others are ignored.
	 */
	createSubProcessor(file, remotePath, defaultSubWorkspaceDir) {
		const suffixMatches = this.config.get('subDirs').filter(
			suffix => file.name.endsWith(suffix)
		);
		if (suffixMatches.length) {
			const suffix = suffixMatches[0];
			if (this.suffixSeen[suffix]) {
				winston.debug(`Ignoring old file ${file.name} (${suffix} already seen)`);
				return null;
			} else {
				this.suffixSeen[suffix] = true;
				// some suffixes are renamed in local copy (planNearFuture -> nearFuture)
				const localSuffix = this.config.get('localSubDirs')[suffix] || suffix;

				return new this.subProcessorClass(
					this.config,
					path.join(this.workspaceDir, localSuffix),
					this.sftpAgent,
					remotePath,
					file,
					{
						...this.logMeta,
						suffix,
					}
				);
			}
		} else {
			winston.debug(`File ${file.name} doesn't have supported suffix`);
			return null;
		}
	}

	sortFiles(files) {
		// alphabetically, descending order (process newest files first)
		files.sort((a, b) => b.name.localeCompare(a.name));
	}

	get taskName() {
		return this.logMeta.topDir;
	}
}

class PolyLevel2DirectoryProcessor extends Level2DirectoryProcessor {
	subProcessorClass = PolyDirectoryProcessor;
}

class RasterLevel2DirectoryProcessor extends Level2DirectoryProcessor {
	subProcessorClass = RasterDirectoryProcessor;
}

/**
 * Processor for poly dirs - just downloads the whole directory.
 */
class PolyDirectoryProcessor extends BaseFileProcessor {
	get taskName() {
		return this.logMeta.suffix;
	}

	async run() {
		const age = fileAge(this.file);
		winston.info(`Downloading poly directory ${this.remotePath} to ${this.workspaceDir}`, {
			...this.logMeta,
			label: 'downloadDir',
			remotePath: this.remotePath,
			localPath: this.workspaceDir,
			age,
		});
		await this.sftpAgent.downloadDir(this.remotePath, this.workspaceDir);
		return {
			remotePath: this.remotePath,
			localPath: this.workspaceDir,
			age,
		}
	}
}

/**
 * Processor for raster dirs - downloads files that match configured regex (^cvgClean.* by default),
 * ignores everything else.
 */
class RasterDirectoryProcessor extends RecursiveDirectoryProcessor {
	get taskName() {
		return this.logMeta.suffix;
	}

	createSubProcessor(file, remotePath, defaultSubWorkspaceDir) {
		if (file.name.match(this.config.get('rasterFileRegExp'))) {
			return new DownloadFileProcessor(
				this.config,
				this.workspaceDir,
				this.sftpAgent,
				remotePath,
				file,
				{
					...this.logMeta,
					fileType: 'raster',
				},
			);
		} else {
			return null;
		}
	}
}

/**
 * Downloads the file.
 */
class DownloadFileProcessor extends BaseFileProcessor {
	get taskName() {
		return this.file.name;
	}

	async run() {
		const localPath = path.join(this.workspaceDir, this.file.name);
		const age = fileAge(this.file);
		winston.info(`Downloading ${this.logMeta.fileType} file ${this.remotePath} to ${this.workspaceDir}`, {
			...this.logMeta,
			label: 'downloadFile',
			remotePath: this.remotePath,
			localPath,
			age,
		});
		await this.sftpAgent.fastGet(this.remotePath, localPath);
		return {
			remotePath: this.remotePath,
			localPath,
			age,
		}
	}
}

function fileAge(remoteFile) {
	return Math.round((Date.now() - remoteFile.modifyTime) / 1000);
}
